/**
 * \file: exchnd_process.c
 *
 * Processing and watchdog functionalities.
 *
 * \component: exchndd
 *
 * \author: Frederic Berat (fberat@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#include <errno.h>
#include <pthread.h>
#include <signal.h>

#include <sys/ioctl.h>
#include <sys/prctl.h>
#include <sys/reboot.h>
#include <sys/syscall.h>

#include "exchnd_backend.h"
#include "exchnd_collector.h"
#include "exchndd.h"

#define EXCH_TH_STARTING    0
#define EXCH_TH_RUNNING     1
#define EXCH_TH_EXITING     2
static int process_th_state;
static int creation_type;

static pthread_t exchnd_watchdog_th;
static pthread_t process_th;

static struct exchnd_process_t {
    pthread_t *th;
    ExchndInterface_t *exh_if;
} th_data = {
    .th = &process_th
};

/* Thread vars */
static pthread_cond_t wd_cond = PTHREAD_COND_INITIALIZER;
static pthread_mutex_t wd_lck = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t new_data = PTHREAD_COND_INITIALIZER;
static pthread_mutex_t data_lock = PTHREAD_MUTEX_INITIALIZER;

static int wd_reset;
static int wd_ping;
static int wd_pid;
static unsigned int wd_collector;
static int wd_pth_pid;

static void *exchnd_watchdog(void *arg)
{
    /* let satisfy lint. */
    struct exchnd_process_t *data = (struct exchnd_process_t *)arg;
    pthread_t *th = data->th;
    int efd = 0;
    pthread_condattr_t attr = { 0 };

    if (!data->exh_if || !data->exh_if->efd) {
        exchnd_print_error("No interface available.");
        action = EXCHND_TERMINATE;
        return NULL;
    }

    pthread_condattr_init(&attr);
    pthread_condattr_setclock(&attr, CLOCK_MONOTONIC);
    pthread_cond_init(&wd_cond, &attr);

    efd = data->exh_if->efd;

    pthread_mutex_lock(&wd_lck);
    prctl(PR_SET_NAME, "watchdog", NULL, NULL, NULL);
    wd_reset = 0;

    while (!(action & EXCHND_TERMINATE)) {
        int ret = 0;
        /* Wait until cleanup needed. */
        pthread_cond_wait(&wd_cond, &wd_lck);

        while (!(action & EXCHND_TERMINATE) && !wd_reset) {
            struct timespec timeout;

            clock_gettime(CLOCK_MONOTONIC, &timeout);
            timeout.tv_sec += 5;

            ret = pthread_cond_timedwait(&wd_cond, &wd_lck, &timeout);

            if (ret == ETIMEDOUT) {
                ioctl(efd, IOCTL_EXCHND_DAEMON_READY, &wd_pid);

                exchnd_print_error("Too slow. Canceling processing thread.");
                exchnd_print_error("Sending SIGHUP and hope to get some logs.");
                exchnd_print_error("Process id was %d, handling collector %u.",
                                   wd_pid,
                                   wd_collector);
                syscall(SYS_tgkill, getpid(), wd_pth_pid, SIGHUP);
                sleep(1);
                pthread_cancel(*th);
                timeout.tv_sec += 5;
                ret = pthread_cond_timedwait(&wd_cond, &wd_lck, &timeout);

                if (ret == ETIMEDOUT) {
                    /* That may wake up the main thread. */
                    pthread_cond_signal(&new_data);
                    exchnd_print_error("Cancel failed, auto-destruction.");
                    kill(0, SIGKILL);
                }
            }

            if (wd_ping) {
                usleep(TEN_MSECS);
                ioctl(efd, IOCTL_EXCHND_DAEMON_READY, &wd_pid);
                wd_ping = 0;
            }
        }
    }

    pthread_mutex_unlock(&wd_lck);

    return NULL;
}

void exchnd_ping_driver()
{
    pthread_mutex_lock(&wd_lck);
    wd_reset = 0;
    wd_ping = 1;
    pthread_cond_signal(&wd_cond);
    pthread_mutex_unlock(&wd_lck);
}

void exchnd_touch_wd(int pid, unsigned int collector)
{
    pthread_mutex_lock(&wd_lck);

    if (pid)
        wd_pid = pid;

    wd_collector = collector;
    wd_reset = 0;
    pthread_cond_signal(&wd_cond);
    pthread_mutex_unlock(&wd_lck);
}

static void exchnd_disarm_wd()
{
    pthread_mutex_lock(&wd_lck);
    wd_reset = 1;
    wd_collector = EHM_NONE;
    pthread_cond_signal(&wd_cond);
    pthread_mutex_unlock(&wd_lck);
}

/*PRQA: Lint Message 455: Clean-up func. Mutex locked in exchnd_process_th. */
/*lint -save -e455*/
static void process_th_cleanup(void *arg)
{
    /* let satisfy lint. */
    (void)arg;

    exchnd_disarm_wd();
    exchnd_print_error("Process thread canceled.");

    /* Don't forget signaling. */
    pthread_cond_signal(&new_data);

    process_th_state = EXCH_TH_EXITING;
    pthread_mutex_unlock(&data_lock);
}
/*lint -restore*/

/*
 * \func exchnd_process_th
 *
 * Execute processing in a separated thread. Advantage of this is that
 * we can exit from this thread in order to force detach on traced
 * threads. This is a safeguard, not a fix.
 *
 * \param arg Pointer to a exchnd_process_t structure that contains all needed
 *            information for exchnd_process function to be executed.
 */
/*PRQA: Lint Message 454: Mutex is unlocked in clean-up function l.207 */
/*lint -save -e454*/
static void *exchnd_process_th(void *arg)
{
    struct exchnd_process_t *data = (struct exchnd_process_t *)arg;
    ExchndInterface_t *exh_if;
    int end_process = 0;

    exchnd_prepare_malloc();

    if (!data || !data->exh_if)
        return NULL;

    /* Memory pool */
    if (creation_type == EXH_INIT_PROCESS_TH)
        exchnd_prepare_pool();

    prctl(PR_SET_NAME, "process", NULL, NULL, NULL);

    exh_if = data->exh_if;

    exchnd_disarm_wd();
    wd_pth_pid = syscall(SYS_gettid);

    pthread_mutex_lock(&data_lock);
    process_th_state = EXCH_TH_RUNNING;
    pthread_cleanup_push(process_th_cleanup, NULL);
    pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &end_process);
    pthread_cond_signal(&new_data);
    pthread_cond_wait(&new_data, &data_lock);

    while (!(action & EXCHND_TERMINATE)) {
        /* Processing data */
        end_process = exchnd_process(exh_if);

        if (end_process) {
#ifdef DEBUG_WATCHDOG
            exchnd_print_error("Processing thread done with its collectors.");
#endif
            exchnd_touch_wd(0, EHM_LAST_ELEMENT);
            /* Now exiting thread to force any remaining
             * attached tracee to be detached.
             */
            exchnd_wake_backends();
            sync();

            if (action & EXCHND_SHUTDOWN) {
                /* Power off the system if we are configured to do so. */
                exchnd_print_error("0x%x", RB_POWER_OFF);
                reboot(RB_POWER_OFF);
            }

            break;
        }

        /* Else then wait for next instruction. */
        pthread_cond_signal(&new_data);
        pthread_cond_wait(&new_data, &data_lock);
    }

    exchnd_disarm_wd();

    /* Don't forget signaling. */
    pthread_cond_signal(&new_data);

    process_th_state = EXCH_TH_EXITING;
    pthread_mutex_unlock(&data_lock);

    pthread_cleanup_pop(0);
    pthread_exit(NULL);
    /* Needed for lint */
    return NULL;
}
/*lint -restore*/

int exchnd_init_processing(ExchndInterface_t *exh_if)
{
    int ret;
    creation_type = EXH_INIT_PROCESS_TH;

    th_data.exh_if = exh_if;

    ret = exchnd_create_thread(&exchnd_watchdog_th,
                               exchnd_watchdog,
                               (void *)&th_data,
                               EXH_NORMAL_TH);

    if (ret) {
        exchnd_print_error("Unable to start watchdog: %s", strerror(errno));
        return ret;
    }

    process_th_state = EXCH_TH_EXITING;
    pthread_mutex_lock(&data_lock);
    ret = exchnd_create_thread(&process_th,
                               exchnd_process_th,
                               &th_data,
                               EXH_INIT_PROCESS_TH);
    pthread_cond_wait(&new_data, &data_lock);
    pthread_mutex_unlock(&data_lock);

    if (ret) {
        exchnd_print_error("Unable to start watchdog: %s", strerror(errno));
        exchnd_deinit_processing();
    }

    return ret;
}

void exchnd_deinit_processing(void)
{
    /* Stop the processing thread */
    pthread_mutex_lock(&data_lock);

    if (process_th_state != EXCH_TH_EXITING) {
        pthread_cond_signal(&new_data);
        pthread_cond_wait(&new_data, &data_lock);
    }

    pthread_mutex_unlock(&data_lock);

    /* Now destroy watchdog */
    pthread_mutex_lock(&wd_lck);
    pthread_cond_signal(&wd_cond);
    pthread_mutex_unlock(&wd_lck);
    pthread_join(exchnd_watchdog_th, NULL);
}

int exchnd_handle_event(void)
{
    int ret = 0;
    creation_type = EXH_PROCESS_TH;
    pthread_mutex_lock(&data_lock);

    if (process_th_state == EXCH_TH_STARTING)
        pthread_cond_wait(&new_data, &data_lock);

    /* process data from /dev/exchnd */
    pthread_cond_signal(&new_data);
    pthread_cond_wait(&new_data, &data_lock);

    if (action & EXCHND_TERMINATE) {
        pthread_mutex_unlock(&data_lock);
        return ret;
    }

    /* Recreate thread if it exited. */
    if (process_th_state == EXCH_TH_EXITING) {
        int limit = 100;

        while ((pthread_kill(process_th, 0) == 0) && limit--)
            /* Wait for the process to finish exit. */
            usleep(100);

        process_th_state = EXCH_TH_STARTING;

        /* init to default values */
        ret = exchnd_create_thread(&process_th,
                                   exchnd_process_th,
                                   &th_data,
                                   EXH_PROCESS_TH);

        if (ret)
            exchnd_print_error("Failed to handle event %s", strerror(errno));
    }

    pthread_mutex_unlock(&data_lock);

    return ret;
}
